﻿using Microsoft.Xrm.Sdk;
using Microsoft.Xrm.Sdk.Messages;
using Microsoft.Xrm.Sdk.Metadata;
using Microsoft.Xrm.Sdk.Query;
using System;
using System.Linq;
using System.ServiceModel;
using System.Text.RegularExpressions;
using PersonSearch.Plugin.Messages;
using System.Diagnostics;
using PersonSearch.Plugin.VistA;
using Xrm;
using System.Security;
using System.Runtime.InteropServices;
using PersonSearch.Plugin.Helpers;
using MVI.PersonSearch.Plugin.Helpers;
using System.Net;
using System.Timers;
using System.IO;
using System.Web.Script.Serialization;
using System.Collections.Generic;

///Updates: 7/17/2017 - Adding logic to allow VCL sensitivity/masking approach to still work while also allowing
///             the other projects to have their 'lighter' version of the masking and sensitivity approaches. 
///         8/17/2017 - Added calls to the new MVILog and MVISensitivityLog services to retain the general 
///             MVI request/response logs as well as the Sensitivity Logs.
///         
namespace PersonSearch.Plugin.Entities.Person
{
    public class VAPersonRetrieveMultiplePostStageRunner : Runner 
    {
        public VAPersonRetrieveMultiplePostStageRunner(IServiceProvider serviceProvider)
            : base(serviceProvider)
        {
        }

        private const string Query = "Query";
        string _uri = "";
        string _orgOverride = "";
        string orgName = "";
        string _requestMessageTypePrefix = "";
        bool _logSoap = false;
        bool _logTimer = false;
        LogSettings _logSettings = null;
        Uri uri = null;
        // auditing and other flags
        bool getSensitiveInfo = false;
        bool enableAuditing = false;
        //bool isVeteran = false;
        string auditUrl = "";
        string userName = "";
        // vba sensitivity
        string stationNumber = "";
        string userSensLevel = "";
        string cssName = "";
      //  bool doNotRun = false;

        internal void Execute()
        {
            try {
                Logger.setDebug = true;
             
                if (!PluginExecutionContext.InputParameters.Contains(Query) ||
                    !(PluginExecutionContext.InputParameters[Query] is QueryExpression)) return;

                try
                {
                    getSettingValues();

                    var qe = (QueryExpression)PluginExecutionContext.InputParameters[Query];

                    var mviSearchType = TryGetSearchType(qe);

                    if (!string.IsNullOrEmpty(mviSearchType))
                    {
                        ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Clear();

                        orgName = (_orgOverride != "" && _orgOverride != string.Empty) ? _orgOverride : PluginExecutionContext.OrganizationName;

                        // set up the settings
                        uri = new Uri(_uri);
                        _logSettings = new LogSettings()
                        {
                            Org = orgName,
                            ConfigFieldName = "RESTCALL",
                            UserId = PluginExecutionContext.InitiatingUserId,
                            callingMethod = "MVIPersonSearchPlugin"
                        };

                        _requestMessageTypePrefix = setMessageType(orgName);
                        string _requestMessageType = "";

                  //      Logger.WriteDebugMessage("DEBUG::Running " + mviSearchType);

                        //we do the same thing regardless of what kind of search, let the VIMT message handle what that means.
                        switch (mviSearchType)
                        {
                            case "SearchByIdentifier":
                            case "SearchByFilter":
                                var SearchRequest = new PersonSearchRequest();
                                SetQueryString((PersonSearchRequest)SearchRequest, qe);
                                SearchRequest.MessageId = Guid.NewGuid().ToString();
                                SearchRequest.OrganizationName = orgName;
                                SearchRequest.UserId = PluginExecutionContext.InitiatingUserId;
                                SearchRequest.Debug = false;
                                SearchRequest.LogSoap = _logSoap;
                                SearchRequest.LogTiming = _logTimer;

                                _requestMessageType = string.Format("{0}PersonSearchRequest", _requestMessageTypePrefix);

                                var SearchResponse = Utility.SendReceive<PersonSearchResponse>(uri, _requestMessageType, SearchRequest, _logSettings);

                                if (SearchResponse.ExceptionOccured)      {
                                    Logger.WriteDebugMessage(string.Format("ERROR::" + mviSearchType + ": Search Exception Message: {0} MVI Message: {1}", SearchResponse.MVIMessage, SearchResponse.RawMviExceptionMessage));
                                }

                                var personFound = ((SearchResponse.Person != null) && (SearchResponse.Person.Length > 0)) ? true : false;

                                Map((PersonSearchResponse)SearchResponse);

                                Logger.WriteDebugMessage(string.Format("DEBUG::" + mviSearchType + ": Search Response Message: {0} Person Found: {1}", SearchResponse.MVIMessage, personFound));
                                break;

                            case "SelectedPersonSearch":
                                var selectedRequest = new SelectedPersonRequest();
                                SetQueryString((SelectedPersonRequest)selectedRequest, qe);
                                selectedRequest.OrganizationName = orgName; 
                                selectedRequest.UserId = PluginExecutionContext.InitiatingUserId;
                                selectedRequest.MessageId = Guid.NewGuid().ToString();
                                selectedRequest.Debug = false;
                                selectedRequest.LogSoap = _logSoap;
                                selectedRequest.LogTiming = _logTimer;
                                selectedRequest.noAddPerson = false;

                                _requestMessageType = string.Format("{0}SelectedPersonRequest", _requestMessageTypePrefix);

                                var selectedResponse = Utility.SendReceive<CorrespondingIdsResponse>(uri, _requestMessageType, selectedRequest, _logSettings);

                                if (selectedResponse.ExceptionOccured)       {
                                    Logger.WriteDebugMessage(string.Format("ERROR::" + mviSearchType + ": Search Exception Message: {0}", selectedResponse.RawMviExceptionMessage));
                                }

                                var correspondingFound = ((selectedResponse.CorrespondingIdList != null) && (selectedResponse.CorrespondingIdList.Length > 0)) ? true : false;
                               
                                MapCorrespondingIds(selectedResponse, selectedRequest);
                              
                                Logger.WriteDebugMessage(string.Format("DEBUG::" + mviSearchType + ": Search Response Message: Corresponding IDs Found: {0}", correspondingFound));
                              
                                break;

                            case "DeterministicSearch":
                                var deterministicRequest = new DeterministicSearchRequest();
                                SetQueryString((DeterministicSearchRequest)deterministicRequest, qe);
                                deterministicRequest.OrganizationName = orgName;
                                deterministicRequest.UserId = PluginExecutionContext.InitiatingUserId;
                                deterministicRequest.MessageId = Guid.NewGuid().ToString();

                                _requestMessageType = string.Format("{0}DeterministicSearchRequest", _requestMessageTypePrefix);
     
                                var deterministicResponse = Utility.SendReceive<PersonSearchResponse>(uri, _requestMessageType, deterministicRequest, _logSettings);
                                            
                                if (deterministicResponse.ExceptionOccured)      {
                                    Logger.WriteDebugMessage(string.Format("ERROR::" + mviSearchType + ": Search Exception Message: {0} MVI Message: {1}", deterministicResponse.MVIMessage, deterministicResponse.RawMviExceptionMessage));
                                }
   
                                var peopleFound = ((deterministicResponse.Person != null) && (deterministicResponse.Person.Length > 0)) ? true : false;
                                                         
                                Map((PersonSearchResponse)deterministicResponse);
                            
                                Logger.WriteDebugMessage(string.Format("DEBUG::" + mviSearchType + ": Search Response Message: {0} Person Found: {1}", deterministicResponse.MVIMessage, peopleFound));
                                
                                break;
                        }
                    }
                }
                catch (Exception ex)
                {
                    Logger.WriteToFile(ex.Message);
                    Logger.WriteDebugMessage(string.Format("ERROR:: Search Exception Occurred. Error Message: {0}", ex.Message));
                    Logger.WriteDebugMessage(string.Format("ERROR:: Search Exception Stack Trace: {0}", ex.StackTrace));

                    // send back a proper message so MVI UI knows to display an error
                    HandleMVIErrorInUI();
                }
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException("Unable to process MVI request due to: {0}".Replace("{0}",
                      ex.Message));
            }
            catch (Exception ex)
            {
                Logger.WriteToFile(ex.Message);
                throw new InvalidPluginExecutionException("Unable to process MVI request due to {0}".Replace("{0}",
                      ex.Message));
            }

           // Logger.WriteDebugMessage("DEBUG::Successful Exit");
        }
        
        #region utility methods

        private string setMessageType(string organization)
        {
            if (organization == null) return "";

            if (organization.ToUpper().Contains("FTP"))    {
                return "FTP";
            }
            else if (organization.ToUpper().Contains("VCL"))    {
                return "VCL";
            }
            else if (organization.ToUpper().Contains("VRE"))    {
                return "VRE";
            }
            else if (organization.ToUpper().Contains("HRC") || organization .ToUpper().Contains("VHA"))       {
                return "HRC";
            }
            else if (organization.ToUpper().Contains("CCWF"))    {
                return "CCWF";
            }
            else if (organization.ToUpper().Contains("ICP"))     {
                return "ICP";
            }
            else if (organization.ToUpper().Contains("FCMT"))    {
                return "FCMT";
            }
            else if (organization.ToUpper().Contains("ECC"))     {
                return "ECC";
            }
            else if (organization.ToUpper().Contains("VEMS"))    {
                return "VEMS";
            }

             return "";
        }

        internal void getSettingValues()
        {
            QueryByAttribute query = new QueryByAttribute
            {
                //change these fields as required for your implementation
                ColumnSet = new ColumnSet("mcs_transactiontiming", "crme_restendpointforvimt", "mcs_granulartimings", "vre_searchlogtiming", "vre_searchlogsoap", "mcs_unexpectedmessage", "bah_masksensitiveinfo", "bah_retrievesensitivitylevel", "bah_enableauditing"),
                EntityName = "mcs_setting"
            };
            query.AddAttributeValue("mcs_name", "Active Settings");

            EntityCollection results = OrganizationService.RetrieveMultiple(query);
            if (results.Entities.Count > 0)
            {
                // not used
                //bool _transTiming = (Boolean)results.Entities[0]["mcs_transactiontiming"];
                //bool _granTimings = (Boolean)results.Entities[0]["mcs_granulartimings"];
                //"mcs_settingid"	string

                // maskInfo = (Boolean)results.Entities[0]["bah_masksensitiveinfo"];
                getSensitiveInfo = (Boolean)results.Entities[0]["bah_retrievesensitivitylevel"]; 
                enableAuditing = (Boolean)results.Entities[0]["bah_enableauditing"];

                _orgOverride = (results.Entities[0].Contains("mcs_unexpectedmessage")) ? results.Entities[0]["mcs_unexpectedmessage"].ToString() : string.Empty;
                _uri = (results.Entities[0].Contains("crme_restendpointforvimt")) ? results.Entities[0]["crme_restendpointforvimt"].ToString() : string.Empty;
                _logTimer = (results.Entities[0].Contains("vre_searchlogtiming")) ? (Boolean)results.Entities[0]["vre_searchlogtiming"] : false;
                _logSoap = (results.Entities[0].Contains("vre_searchlogsoap")) ? (Boolean)results.Entities[0]["vre_searchlogsoap"] : false;
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="qe"></param>
        /// <returns></returns>
        private string TryGetSearchType(QueryExpression qe)
        {
            try
            {
                var searchType =
                    qe.Criteria.Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));

                searchType = qe.Criteria.Conditions.FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                
                // the remaining code will handle the standard QueryExpression, 
                // the above code deals with fetchXML specific queries

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }

                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }
                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }
                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }
                if (searchType == null)
                {
                    searchType = qe.Criteria.Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters[0].Filters.SelectMany(v => v.Conditions)
                        .FirstOrDefault(v => v.AttributeName
                            .Equals("crme_SearchType", StringComparison.OrdinalIgnoreCase));
                }
                if (searchType == null) return string.Empty;

                var firstOrDefault = searchType.Values.FirstOrDefault();
                return firstOrDefault != null ? firstOrDefault.ToString() : string.Empty;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="aliasName"></param>
        /// <returns></returns>
        private static string TryGetAlias(Name aliasName)
        {
            try
            {
                if (aliasName == null)
                {
                    return string.Empty;
                }

                const string nameFormat = "{0} {1}";
                var alias = string.Format(nameFormat, aliasName.GivenName, aliasName.FamilyName);

                return alias;
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="person"></param>
        /// <returns></returns>
        private static void TryGetMviQueryParams(PatientPerson person, crme_person newPerson, Logger Logger)
        {
            try
            {
                if (person.CorrespondingIdList == null || !person.CorrespondingIdList.Any())
                {
                    newPerson.crme_PatientMviIdentifier = string.Empty;
                    newPerson.crme_ICN = string.Empty;
                }
                else
                {

                    var patientNo = person.CorrespondingIdList.FirstOrDefault((v =>
                        v.AssigningAuthority != null &&
                        v.AssigningAuthority.Equals("USVHA", StringComparison.InvariantCultureIgnoreCase) &&
                        v.AssigningFacility != null && v.AssigningFacility == "200M" &&
                        v.IdentifierType != null &&
                        v.IdentifierType.Equals("NI", StringComparison.InvariantCultureIgnoreCase))) ??
                                    person.CorrespondingIdList.FirstOrDefault((v =>
                                        v.AssigningAuthority != null &&
                                        v.AssigningAuthority.Equals("USVHA", StringComparison.InvariantCultureIgnoreCase) &&
                                        v.IdentifierType != null &&
                                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase)));

                    newPerson.crme_PatientMviIdentifier = patientNo != null ? patientNo.RawValueFromMvi : string.Empty;
                    newPerson.crme_ICN = patientNo != null ? patientNo.PatientIdentifier : string.Empty;
                    //newPerson.crme_ParticipantID = patientNo != null ? patientNo.PatientIdentifier : string.Empty;
                }
            }
            catch (ArgumentNullException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (NullReferenceException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="person"></param>
        /// <returns></returns>
        private static void TryGetCorpInfo(PatientPerson person, crme_person newPerson, Logger Logger)
        {
            try
            {
                if (person.CorrespondingIdList == null || !person.CorrespondingIdList.Any())
                {
                    newPerson.crme_ParticipantID = string.Empty;
                }
                else
                {

                    var patientNo = person.CorrespondingIdList.FirstOrDefault((v =>
                        v.AssigningAuthority != null &&
                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                        v.AssigningFacility != null && v.AssigningFacility == "200CORP" &&
                        v.IdentifierType != null &&
                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase))) ??
                                    person.CorrespondingIdList.FirstOrDefault((v =>
                                        v.AssigningAuthority != null &&
                                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                                        v.IdentifierType != null &&
                                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase)));

                    newPerson.crme_ParticipantID = patientNo != null ? patientNo.PatientIdentifier : string.Empty;    
                  //  newPerson.crme_FileNumber = patientNo != null ? patientNo.PatientIdentifier : string.Empty;                 
                }
            }
            catch (ArgumentNullException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (NullReferenceException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        private static void TryGetBIRLSInfo(PatientPerson person, crme_person newPerson, Logger Logger)
        {
            try
            {
                if (person.CorrespondingIdList == null || !person.CorrespondingIdList.Any())
                {
                    newPerson.crme_FileNumber = string.Empty;
                }
                else
                {
                    //992003^PI^200BRLS^USVBA
                    var patientNo = person.CorrespondingIdList.FirstOrDefault((v =>
                        v.AssigningAuthority != null &&
                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                        v.AssigningFacility != null && v.AssigningFacility == "200BRLS" &&
                        v.IdentifierType != null &&
                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase))) ??
                                    person.CorrespondingIdList.FirstOrDefault((v =>
                                        v.AssigningAuthority != null &&
                                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                                        v.IdentifierType != null &&
                                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase)));

                    newPerson.crme_FileNumber = patientNo != null ? patientNo.PatientIdentifier : string.Empty;                                   
                }
            }
            catch (ArgumentNullException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (NullReferenceException ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get MVI Query Params due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        private static void TryGetEDIPI(PatientPerson person, crme_person newPerson, Logger Logger)
        {
            try
            {
                if (person.CorrespondingIdList == null || !person.CorrespondingIdList.Any())
                {
                    newPerson.crme_EDIPI = string.Empty;
                }
                else
                {
                    var patientNo = person.CorrespondingIdList.FirstOrDefault((v =>
                        v.AssigningAuthority != null &&
                        v.AssigningAuthority.Equals("USDOD", StringComparison.InvariantCultureIgnoreCase) &&
                        v.AssigningFacility != null && v.AssigningFacility == "200DOD" &&
                        v.IdentifierType != null &&
                        v.IdentifierType.Equals("NI", StringComparison.InvariantCultureIgnoreCase))) ??
                                    person.CorrespondingIdList.FirstOrDefault((v =>
                                        v.AssigningAuthority != null &&
                                        v.AssigningAuthority.Equals("USDOD", StringComparison.InvariantCultureIgnoreCase) &&
                                        v.IdentifierType != null &&
                                        v.IdentifierType.Equals("NI", StringComparison.InvariantCultureIgnoreCase)));

                    newPerson.crme_EDIPI = patientNo != null ? patientNo.PatientIdentifier : string.Empty;
                }
            }
            catch (ArgumentNullException ex)
            {
                Logger.WriteToFile("Unable to Get EDIPI due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get EDIPI due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteToFile("Unable to Get EDIPI due to: {0}".Replace("{0}",
                   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get EDIPI due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (NullReferenceException ex)
            {
                Logger.WriteToFile("Unable to Get EDIPI due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get EDIPI due to: {0}".Replace("{0}",
                    ex.Message));
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to Get EDIPI due to: {0}".Replace("{0}",
   ex.Message));
                throw new InvalidPluginExecutionException("Unable to Get EDIPI due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        #region required overrides
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override Entity GetPrimaryEntity()
        {
            return (Entity)PluginExecutionContext.InputParameters["Target"];
        }
        /// <summary>
        /// 
        /// </summary>
        /// <returns></returns>
        public override Entity GetSecondaryEntity()
        {
            return (Entity)PluginExecutionContext.InputParameters["Target"];
        }

        //public override string McsSettingsDebugField
        //{
        //    get { return string.Empty; }
        //}
        #endregion

        /// <summary>
        /// 
        /// </summary>
        ///// <param name="userId"></param>
        /// <returns></returns>
        private Entity TryGetUserInfo(Guid userId)
        {
            Entity user = new Entity("systemuser");
            QueryByAttribute query = null;

            // will vary here for VBA vs. VHA applications
            if (orgName.ToUpper().Contains("VRE"))
            {
                query = new QueryByAttribute     {
                    ColumnSet = new ColumnSet("firstname", "lastname", "vre_css_text", "new_sensitivitylevel", "vre_stationid"),
                    EntityName = "systemuser"
                };
            }
            else   {
                query = new QueryByAttribute     {
                    ColumnSet = new ColumnSet("firstname", "lastname"),
                    EntityName = "systemuser"
                };
            }

            Entity result = OrganizationService.Retrieve(query.EntityName, userId, query.ColumnSet);

            if (result != null)     {
                user["FirstName"] = result.Attributes.Contains("firstname") && result.Attributes["firstname"] != null ? result.Attributes["firstname"].ToString() : string.Empty;
                user["LastName"] = result.Attributes.Contains("lastname") && result.Attributes["lastname"] != null ? result.Attributes["lastname"].ToString() : string.Empty;
            }

            if (orgName.ToUpper().Contains("VRE"))     {
                if (result.Attributes.Contains("vre_stationid"))      {
                    EntityReference station = (EntityReference)result.Attributes["vre_stationid"];
                    if (station != null)     {
                        int start = station.Name.IndexOf("(") + 1;
                        int len = station.Name.IndexOf(")") - start;
                        stationNumber = station.Name.Substring(start, len);                       
                    }
                }
                cssName = result.Attributes.Contains("vre_css_text") && result.Attributes["vre_css_text"] != null ? result.Attributes["vre_css_text"].ToString() : string.Empty;
                userSensLevel = result.Attributes.Contains("new_sensitivitylevel") && result.Attributes["new_sensitivitylevel"] != null ? result.Attributes["new_sensitivitylevel"].ToString() : string.Empty;
            }

            return user;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="request"></param>
        /// <param name="qe"></param>
        public void SetQueryString(PersonSearchRequest request, QueryExpression qe)
        {
            try
            {
                Logger.setMethod = "SetQueryString";

                if (qe.Criteria != null)
                {
                    if (qe.Criteria.Filters.Any())
                    {
                        request.Edipi = GetStringValueOrDefault(qe.Criteria, "crme_edipi");
                        request.FirstName = GetStringValueOrDefault(qe.Criteria, "crme_firstname");           
                        request.MiddleName = GetStringValueOrDefault(qe.Criteria, "crme_middlename");
                        request.FamilyName = GetStringValueOrDefault(qe.Criteria, "crme_lastname");
                        request.BirthDate = GetStringValueOrDefault(qe.Criteria, "crme_dobstring");
                        request.PhoneNumber = GetStringValueOrDefault(qe.Criteria, "crme_primaryphone");     
                        var isAttended = GetStringValueOrDefault(qe.Criteria, "crme_isattended");
                        request.IsAttended = Convert.ToBoolean(string.IsNullOrEmpty(isAttended) ? "false" : isAttended);
                        request.OrganizationName = PluginExecutionContext.OrganizationName;
                        request.FetchMessageProcessType = "remote";
                        request.UserId = PluginExecutionContext.UserId;
                        Entity user = TryGetUserInfo(PluginExecutionContext.UserId);
 
                        if (user != null)   {
                            request.UserFirstName = user["FirstName"].ToString();
                            request.UserLastName = user["LastName"].ToString();
                            userName = request.UserFirstName + " " + request.UserLastName;
                        }

                        //adding additional search params here...
                        request.Address = GetStringValueOrDefault(qe.Criteria, "crme_fulladdress");
                        request.Gender = GetStringValueOrDefault(qe.Criteria, "crme_gender");
                        request.MMN = GetStringValueOrDefault(qe.Criteria, "crme_mmn");
                        request.POBC = GetStringValueOrDefault(qe.Criteria, "crme_pobc");
                        request.POBS = GetStringValueOrDefault(qe.Criteria, "crme_pobs");

                        request.SocialSecurityNumber = GetStringValueOrDefault(qe.Criteria, "crme_ssn");

                        // adding Auditing
                        if (enableAuditing)   {
                            auditRequest(request);
                           // Logger.WriteDebugMessage(auditLog);
                        }

                        if (request.SocialSecurityNumber.Contains("XXX"))
                        {
                            request.SocialSecurityNumber = "";
                        }
                        if (request.BirthDate.Contains("XX/XX"))
                        {
                            request.BirthDate = "";                            
                        }          
                    }
                    else if (qe.Criteria.Conditions.Any())
                    {
                        request.Edipi = GetStringValueOrDefaultFetch(qe.Criteria, "crme_edipi");
                        request.FirstName = GetStringValueOrDefaultFetch(qe.Criteria, "crme_firstname");
                        request.MiddleName = GetStringValueOrDefaultFetch(qe.Criteria, "crme_middlename");
                        request.FamilyName = GetStringValueOrDefaultFetch(qe.Criteria, "crme_lastname");
                        request.BirthDate = GetStringValueOrDefaultFetch(qe.Criteria, "crme_dobstring");
                        request.PhoneNumber = GetStringValueOrDefaultFetch(qe.Criteria, "crme_primaryphone");
                        var isAttended = GetStringValueOrDefaultFetch(qe.Criteria, "crme_isattended");
                        request.IsAttended = Convert.ToBoolean(string.IsNullOrEmpty(isAttended) ? "false" : isAttended);
                        request.OrganizationName = PluginExecutionContext.OrganizationName;
                        request.FetchMessageProcessType = "remote";
                        request.UserId = PluginExecutionContext.UserId;
                        Entity user = TryGetUserInfo(PluginExecutionContext.UserId);

                        if (user != null)   {
                            request.UserFirstName = user["FirstName"].ToString();
                            request.UserLastName = user["LastName"].ToString();
                            userName = request.UserFirstName + " " + request.UserLastName;
                        }

                        //adding additional search params here...     
                        request.Address = GetStringValueOrDefaultFetch(qe.Criteria, "crme_fulladdress");
                        request.Gender = GetStringValueOrDefaultFetch(qe.Criteria, "crme_gender");
                        request.MMN = GetStringValueOrDefaultFetch(qe.Criteria, "crme_mmn");
                        request.POBC = GetStringValueOrDefaultFetch(qe.Criteria, "crme_pobc");
                        request.POBS = GetStringValueOrDefaultFetch(qe.Criteria, "crme_pobs");
                        request.SocialSecurityNumber = GetStringValueOrDefaultFetch(qe.Criteria, "crme_ssn");

                        // adding Auditing
                        if (enableAuditing)    {
                            auditRequest(request);
                            //Logger.WriteDebugMessage(auditLog);
                        }
                    }
                }

                Logger.setMethod = "Execute";
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to set Query String due to: {0}".Replace("{0}",
                    ex.Message));
                throw new InvalidPluginExecutionException("Unable to set Query String due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        /// <summary>
        /// Will handle adding the request details into a string to be kept in a log record
        /// </summary>
        /// <param name="request">PersonSearchRequest object</param>
        /// <returns>string of the requet</returns>
        private void auditRequest(PersonSearchRequest request)
        {
            if (auditUrl == "")
                auditUrl = getKVPSetting("MVI_Log");

            bool attended = request.IsAttended;

            var Log = new MVILog          {
                Address = request.Address,
                Attended = attended,
                ClassCode = request.IdentifierClassCode,
                CreatedOn = DateTime.UtcNow,
                DOB = request.BirthDate,
                EDIPI = request.Edipi,
                First = request.FirstName,
                ICN = request.PatientIdentifier,
                Last = request.FamilyName,
                Middle = request.MiddleName,
                MMN = request.MMN,
                Organization_Name = orgName,
                Phone = request.PhoneNumber,
                POBC = request.POBC,
                POBS = request.POBS,
                RequestType = "PersonSearchRequest",
                SessionId = new Guid().ToString(),
                Sex = request.Gender,
                Source = "",
                SSN = request.SocialSecurityNumber,
                Type = "Request",
                UserName = userName
            };

            var json = new JavaScriptSerializer().Serialize(Log);
            handleRequest(auditUrl, json, "MVILog");
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="request"></param>
        /// <param name="qe"></param>
        public void SetQueryString(SelectedPersonRequest request, QueryExpression qe)
        {

           /// Logger.WriteDebugMessage("DEBUG::Setting query string...DoNotRun:" + doNotRun);
            try
            {
                Logger.setMethod = "SetQueryString";         

                if (qe.Criteria != null)       {
                    if (qe.Criteria.Filters.Any())       {                 
                        request.Edipi = GetStringValueOrDefault(qe.Criteria, "crme_edipi");
                        request.ICN = GetStringValueOrDefault(qe.Criteria, "crme_ICN");
                        request.FirstName = GetStringValueOrDefault(qe.Criteria, "crme_firstname");
                        request.MiddleName = GetStringValueOrDefault(qe.Criteria, "crme_middlename");
                        request.FamilyName = GetStringValueOrDefault(qe.Criteria, "crme_lastname");
                        request.RawValueFromMvi = GetStringValueOrDefault(qe.Criteria, "crme_PatientMviIdentifier");
                        request.RecordSource = GetStringValueOrDefault(qe.Criteria, "crme_RecordSource");

                        if (request.RecordSource == "MVI")       {
                            request.DateofBirth = GetStringValueOrDefault(qe.Criteria, "crme_MVIDOBString");
                            request.FullAddress = GetStringValueOrDefault(qe.Criteria, "crme_MVIFullAddress");
                            request.FullName = GetStringValueOrDefault(qe.Criteria, "crme_MVIFullName");
                        }
                        request.OrganizationName = PluginExecutionContext.OrganizationName;
                        request.FetchMessageProcessType = "remote";
                        request.UserId = PluginExecutionContext.UserId;

                        if (!string.IsNullOrEmpty(request.Edipi))        {
                            request.IdentifierClassCode = "MIL";
                        }

                        Entity user = TryGetUserInfo(PluginExecutionContext.UserId);
                        if (user != null)       {
                            request.UserFirstName = user["FirstName"].ToString();
                            request.UserLastName = user["LastName"].ToString();
                            userName = request.UserFirstName + " " + request.UserLastName;
                        }

                        request.SocialSecurityNumber = GetStringValueOrDefault(qe.Criteria, "crme_ssn");

                        // auditing response
                        if (enableAuditing)   {
                            auditRequest(request);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to set Query String due to: {0}".Replace("{0}",
                      ex.Message));
                throw new InvalidPluginExecutionException("Unable to set Query String due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        /// <summary>
        /// Will handle adding the request details into a string to be kept in a log record
        /// </summary>
        /// <param name="request">SelectedPersonRequest object</param>
        /// <returns>string of the requet</returns>
        private void auditRequest(SelectedPersonRequest request)
        {
            if (auditUrl == "")
                auditUrl = getKVPSetting("MVI_Log2");

            var Log = new MVILog2()
            {
                Organization_Name = request.OrganizationName,
                CreatedOn = DateTime.UtcNow,
                DOB = request.DateofBirth,
                EDIPI = request.Edipi,                
                ICN = request.ICN,
                RequestType = "SelectedPersonRequest",
                SessionId = new Guid().ToString(),
                SSN = request.SocialSecurityNumber,
                UserName = userName
            };

            var json = new JavaScriptSerializer().Serialize(Log);
            handleRequest(auditUrl, json, "MVILog2");
        }

        /// <summary>
        /// 
        /// </summary>
        /// <param name="request"></param>
        /// <param name="qe"></param>
        public void SetQueryString(DeterministicSearchRequest request, QueryExpression qe)
        {
            try
            {
                Logger.setMethod = "SetQueryString";

                if (qe.Criteria != null)
                {
                    if (qe.Criteria.Filters.Any())
                    {
                        request.EdiPi = GetStringValueOrDefault(qe.Criteria, "crme_edipi");
                        request.BirthDate = GetStringValueOrDefaultFetch(qe.Criteria, "crme_dobstring");
                        request.SearchType = string.IsNullOrWhiteSpace(request.EdiPi) ? DeterministicSearchType.SocialSecurity : DeterministicSearchType.Edipi;
                        request.UserId = PluginExecutionContext.UserId;

                        // get user info
                        Entity user = TryGetUserInfo(PluginExecutionContext.UserId);
                        if (user != null)    {
                            request.UserFirstName = user["FirstName"].ToString();
                            request.UserLastName = user["LastName"].ToString();
                            userName = request.UserFirstName + " " + request.UserLastName;
                        }                        
                        request.SocialSecurityNumber = GetStringValueOrDefault(qe.Criteria, "crme_ssn");

                        // adding Auditing
                        if (enableAuditing)    {
                            auditRequest(request);
                        }
                    }
                }
            }
            catch (Exception ex)
            {
                Logger.WriteToFile("Unable to set Query String due to: {0}".Replace("{0}",
                      ex.Message));
                throw new InvalidPluginExecutionException("Unable to set Query String due to: {0}".Replace("{0}",
                    ex.Message));
            }
        }

        /// <summary>
        /// Will handle adding the request details into a string to be kept in a log record
        /// </summary>
        /// <param name="request">DeterministicSearchRequest object</param>
        /// <returns>string of the requet</returns>
        private void auditRequest(DeterministicSearchRequest request)
        {
            if (auditUrl == "")
                auditUrl = getKVPSetting("MVI_Log");

            var Log = new MVILog    {
                Address = "",
                Attended = false,
                ClassCode = "",
                CreatedOn = DateTime.UtcNow,
                DOB = request.BirthDate,
                EDIPI = request.EdiPi,
                First = "",
                ICN = "",
                Last = "",
                Middle = "",
                MMN = "",
                Organization_Name = orgName,
                Phone = "",
                POBC = "",
                POBS = "",
                RequestType = "DeterministicSearchRequest", 
                SessionId = new Guid().ToString(),
                Sex = "",
                Source = "",
                SSN = request.SocialSecurityNumber,
                Type = "Request",
                UserName = userName
            };

            var json = new JavaScriptSerializer().Serialize(Log);
            handleRequest(auditUrl, json, "MVILog");
        }

        /// <summary>
        /// /
        /// </summary>
        /// <param name="expression"></param>
        /// <param name="fieldName"></param>
        /// <returns></returns>
        private static string GetStringValueOrDefaultFetch(FilterExpression expression, string fieldName)
        {
            try
            {
                var conditions =
                  expression
                  .Conditions.FirstOrDefault(
                       v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));

                if (conditions != null)
                {
                    return conditions.Values[0].ToString();
                }
            }
            catch (Exception)
            {
                return string.Empty;
            }

            return string.Empty;
        }
        /// <summary>
        /// /
        /// </summary>
        /// <param name="expression"></param>
        /// <param name="fieldName"></param>
        /// <returns></returns>
        private static string GetStringValueOrDefault(FilterExpression expression, string fieldName)
        {
            try
            {
                var conditions =
                    expression
                    .Filters[0]
                    .Conditions.FirstOrDefault(
                         v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Conditions.FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions.FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions.FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions.FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions.FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }

                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }
                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }
                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }
                if (conditions == null)
                {
                    conditions =
                        expression
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Filters[0]
                        .Conditions
                        .FirstOrDefault(
                            v => v.AttributeName.Equals(fieldName, StringComparison.OrdinalIgnoreCase));
                }
                if (conditions != null)
                {
                    return conditions.Values[0].ToString();
                }
            }
            catch (Exception)
            {
                return string.Empty;
            }

            return string.Empty;
        }
        /// <summary>
        /// 
        /// </summary>
        /// <param name="optionSetString"></param>
        /// <param name="entityName"></param>
        /// <param name="attributeName"></param>
        /// <returns></returns>
        public int GetOptionSetValue(string optionSetString, string entityName, string attributeName)
        {
            try
            {
                var attributeRequest = new RetrieveAttributeRequest
                {
                    EntityLogicalName = entityName,
                    LogicalName = attributeName,
                    RetrieveAsIfPublished = false
                };

                // Retrieve only the currently published changes, ignoring the changes that have
                // not been published.

                var attributeResponse = (RetrieveAttributeResponse)OrganizationService.Execute(attributeRequest);

                // Access the retrieved attribute.
                var retrievedAttributeMetadata = (PicklistAttributeMetadata)attributeResponse.AttributeMetadata;
                return (from t in retrievedAttributeMetadata.OptionSet.Options
                        where t.Label.LocalizedLabels[0].Label == optionSetString
                        let value = t.Value
                        where value != null
                        select value.Value).FirstOrDefault();
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.setModule = "getOptionSetValue";
                Logger.WriteToFile(ex.Detail.Message);
                Logger.setModule = "execute";
                return 0;
            }
            catch (Exception ex)
            {
                Logger.setModule = "getOptionSetValue";
                Logger.WriteToFile(ex.Message);
                Logger.setModule = "execute";
                return 0;
            }
        }

        private SecureString ConvertToSecureString(string value)
        {
            if (value == null)
                throw new ArgumentNullException("password");

            var secureValue = new SecureString();

            foreach (char c in value)
                secureValue.AppendChar(c);

            secureValue.MakeReadOnly();
            return secureValue;
        }

        string SecureStringToString(SecureString value)
        {
            IntPtr valuePtr = IntPtr.Zero;
            try
            {
                valuePtr = Marshal.SecureStringToGlobalAllocUnicode(value);
                return Marshal.PtrToStringUni(valuePtr);
            }
            finally
            {
                Marshal.ZeroFreeGlobalAllocUnicode(valuePtr);
            }
        }
         
        internal string getKVPSetting(string name)
        {
            QueryByAttribute query = new QueryByAttribute
            {
                //change these fields as required for your implementation
                ColumnSet = new ColumnSet("bah_stringvalue_text"),
                EntityName = "bah_keyvaluepair"
            };
            query.AddAttributeValue("bah_name_text", name);

            EntityCollection results = OrganizationService.RetrieveMultiple(query);
            if (results.Entities.Count > 0)
            {
                return (results.Entities[0].Contains("bah_stringvalue_text")) ? results.Entities[0]["bah_stringvalue_text"].ToString() : string.Empty;
            }

            return string.Empty;
        }

        /// <summary>
        /// Gets the sensitivity level from ESR
        /// </summary>
        /// <param name="url"></param>
        /// <param name="icn"></param>
        /// <returns></returns>
        private string getSensitivityLevelVHA(string url, string icn)
        {
            url = url.Replace("xml", "json");

            if (url.Contains("{0}"))
                url = url.Replace("{0}", "");

            using (var client = new WebClient())
            {
                var json = client.DownloadString(url + icn);

                var serializer = new JavaScriptSerializer();
                ESRSensitiveObject ro = serializer.Deserialize<ESRSensitiveObject>(json);

                if (ro.Data.EnrollmentDeterminationInfo != null && ro.Data.EnrollmentDeterminationInfo.PrimaryEligibility != null &&
                    ro.Data.EnrollmentDeterminationInfo.PrimaryEligibility.Type != null && ro.Data.EnrollmentDeterminationInfo.PrimaryEligibility.Type.ToUpper() == "EMPLOYEE")
                {
                    return ro.Data.SensitivityInfo.SensityFlag.ToLower() + ":true";
                }
                else
                    return ro.Data.SensitivityInfo.SensityFlag.ToLower() + ":false";
            }
        }

        private string getSensitivityLevelVBA(string url, string pId, bool byPid)
        {
            if (!byPid)
                url = url.Replace("SensitivityByPid", "SensitivityByFileNumber");

           url = url.Replace("xml", "json");
           url = url.Replace("<cssName>", cssName);
           url = url.Replace("<stationNumber>", stationNumber);
           url = url.Replace("<pId>", pId);

           using (var client = new WebClient())
           {
               var json = client.DownloadString(url);

               var serializer = new JavaScriptSerializer();
               SensitiveObject ro = serializer.Deserialize<SensitiveObject>(json);
               return (ro.Data[0].Return1 == null) ? "" : ro.Data[0].Return1.ScrtyLevelTypeCd;
           }
           //Logger.WriteToFile("Sensitivity Level Not Found for PatientId: {0}".Replace("{0}", pId));
           //return "10";
        }

        #endregion

        #region Selected Person Search Mapping

        private void MapCorrespondingIds(CorrespondingIdsResponse response, SelectedPersonRequest request)
        {
            //if (doNotRun)
            //{
            //    Logger.WriteDebugMessage("DEBUG::Do Not Run so exiting..");
            //    return;
            //}
            //Logger.WriteDebugMessage("DEBUG::did not exit..");


            try     {
                var newPerson = new crme_person { Id = Guid.NewGuid(), crme_FullName = string.Empty };

                if (response.ExceptionOccured)     {
                    newPerson.crme_ExceptionOccured = true;
                    newPerson.crme_ExceptionMessage = "An unexpected error occured during the MVI search. Please try again or contact your system administrator if the problem persists.";// response.MVIMessage;
                }

                if (response.CorrespondingIdList == null || !response.CorrespondingIdList.Any())       {
                    Logger.WriteDebugMessage("DEBUG::No Corresponding IDs retrieved.");
                    newPerson.crme_ReturnMessage = response.Message;

                    if (!string.IsNullOrEmpty(response.RawMviExceptionMessage))        {
                        Logger.WriteDebugMessage(response.RawMviExceptionMessage);
                    }

                    ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Add(newPerson);
                    return;
                }

                if (response.ExceptionOccured)    {
                    newPerson.Id = Guid.NewGuid();
                    newPerson.crme_FullName = string.Empty;
                    newPerson.crme_ReturnMessage = response.Message;

                    if (!string.IsNullOrEmpty(response.RawMviExceptionMessage))      {
                        Logger.WriteDebugMessage(response.RawMviExceptionMessage);
                    }

                    ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Add(newPerson);
                    return;
                }
                
                string sensitiveVeteran = string.Empty;

                if (getSensitiveInfo)
                {
                    // get the URL for ESR
                    string url = getKVPSetting("sensitive_endpoint");

                    sensitiveVeteran = getSensitivityLevelVHA(url, request.ICN);
                    // write to the audit log
    //                if (sensitiveVeteran == "true:false" || sensitiveVeteran == "true:true")   // isSensitive : isVeteran
    //                    auditSensitivity(response.FullName, true);
                }
                              
                string ids = string.Empty;
                string siteName = string.Empty;

                //  return all keys
                foreach (var correspondingId in response.CorrespondingIdList)       {
                    var personInstance = new crme_person();
                    Guid ID = new Guid();
                    personInstance.Id = ID;
                    personInstance.crme_personId = ID;
                    personInstance.crme_SiteId = correspondingId.AssigningFacility;

                    #region retain FTP logic
                    if (_requestMessageTypePrefix == "FTP")    { 
                        siteName = string.Empty;
                        try   {
                            siteName = VistaSiteService.GetSiteName(correspondingId.AssigningFacility, OrganizationService);
                        }
                        catch (Exception ex)    {
                           // Logger.WriteDebugMessage("Error::There is no Vista Site entity present in the system.");
                        }
                            
                        if (siteName == string.Empty)
                            siteName = correspondingId.AssigningFacility;
                            
                      //  int i = 0;
                      //  bool result = int.TryParse(siteName, out i);
                        personInstance.crme_SiteName = siteName;// result ? siteName : siteName.Substring(0, siteName.IndexOf("&"));
                        personInstance.crme_url = "http://event/?eventname=Set location&site=" + siteName;
                    }
                    #endregion

                    personInstance.crme_DOBString = FormatDate(response.DateofBirth);
                    personInstance.crme_MVIDOBString = response.DateofBirth;
                    personInstance.crme_PrimaryPhone = response.PhoneNumber;
                    personInstance.crme_FullAddress = response.FullAddress;                        
                    personInstance.crme_FirstName = response.FirstName;
                    personInstance.crme_LastName = response.FamilyName;
                    personInstance.crme_PatientMviIdentifier = request.RawValueFromMvi;
                    personInstance.crme_PatientId = correspondingId.PatientIdentifier;
                    personInstance.crme_FullName = response.FullName;
                    personInstance.crme_VeteranSensitivityLevel = sensitiveVeteran;

                    // adding for logging
                    ids = ids == string.Empty ? request.RawValueFromMvi : ids +  ";" + request.RawValueFromMvi;
                
                    // for Corp/BRLS add the FileNumber
                    if (correspondingId.AssigningFacility.Contains("CORP") && correspondingId.AssigningAuthority == "USVBA" && correspondingId.IdentifierType == "PI")     {
                        personInstance.crme_FileNumber = correspondingId.PatientIdentifier;
                    }

                    // attempt to get the EDIPI, if present
                    if (personInstance.crme_EDIPI == "" && correspondingId.AssigningFacility.Contains("DOD") && correspondingId.AssigningAuthority == "USDOD" && correspondingId.IdentifierType == "NI")      {
                        personInstance.crme_EDIPI = correspondingId.PatientIdentifier;
                    }

                    // moved to bottom for Fortify scans
                    personInstance.crme_SSN = response.SocialSecurityNumber;
                    
                    ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Add(personInstance);
                }
                
                if (enableAuditing)      
                    auditSelectedPersonResponse(response, ids);
                
            }
            catch (FaultException<OrganizationServiceFault> ex)
            {
                Logger.WriteDebugMessage("ERROR::OrganizationServiceFault Exception while mapping PatientIdentifier:" + ex.Message);
            }
            catch (NullReferenceException ex)
            {
                Logger.WriteDebugMessage("ERROR::Null Reference Exception while mapping PatientIdentifier:" + ex.Message);
            }
            catch (Exception ex)
            {
                Logger.WriteDebugMessage("ERROR::Error while mapping PatientIdentifier:" + ex.Message);
            }    
        }

        /// <summary>
        /// Will return a string of the id response
        /// </summary>
        /// <param name="count"></param>
        /// <param name="person"></param>
        /// <returns></returns>
        private void auditSelectedPersonResponse(CorrespondingIdsResponse response, string ids)
        {
            if (auditUrl == "")
                auditUrl = getKVPSetting("MVI_Log2");

            var Log2 = new MVILog2()
            {
                Organization_Name = response.OrganizationName,
                CreatedOn = DateTime.UtcNow,
                DOB = response.DateofBirth,
                EDIPI = response.Edipi,
                ICN = response.ParticipantId,
                IDs = ids,
                Phone = response.PhoneNumber,
                SessionId = new Guid().ToString(),
                SSN = response.SocialSecurityNumber,
                UserName = userName
            };

            var json = new JavaScriptSerializer().Serialize(Log2);
            handleRequest(auditUrl, json, "MVILog2");
        }

        /// <summary>
        /// Will return a string of the id response
        /// </summary>
        /// <param name="count"></param>
        /// <param name="person"></param>
        /// <returns></returns>
        private void auditSensitivity(string crmeFullName, bool accessed)
        {
            string sensitiveUrl = getKVPSetting("MVI_Sensitivity_Log");

            var Log = new SensitiveLog()   {
                Accessed = accessed, 
                Access_Time = DateTime.UtcNow.ToString("G"),
                HDR_Response_Code = "",
                Organization_Name = orgName.Trim(),
                Sensitive_Veteran = true, 
                SessionId = new Guid().ToString(),
                Veteran = crmeFullName,
                Who_Accessed = userName
            };

            var json = new JavaScriptSerializer().Serialize(Log);
            handleRequest(sensitiveUrl, json, "Sensitivity_Log");
        }

        #endregion

        #region Person Search Mapping
        private void Map(PersonSearchResponse response)
        {
            var personCollection = new EntityCollection();
            crme_person newPerson = null; 
            bool setMessage = false;

            // exception check
            if (response.ExceptionOccured)   {                                                                             
                if (response.MVIMessage.ToLower().Contains("correlation does not exist") || response.MVIMessage.ToLower().Contains("unknown key identifier"))   {
                    // Not a real exception. The EDIPI not found throws an application exception in MVI but we just want to report Veteran not found
                    setMessage = true;
                }
                else   {
                    newPerson = new crme_person { Id = Guid.NewGuid(), crme_FullName = string.Empty };
                    newPerson.crme_ExceptionOccured = true;
                    newPerson.crme_ExceptionMessage = "An unexpected error occured during the MVI search. Please try again or contact your system administrator if the problem persists.";
                }
            }

            if (response.Person == null || (response.Person != null && response.Person[0] == null))    {
                if (newPerson == null)  {
                    newPerson = new crme_person { Id = Guid.NewGuid(), crme_FullName = string.Empty };
                }

                if (setMessage)    {
                    newPerson.crme_ExceptionMessage = "Your search in MVI did not find any records matching the search criteria.";
                }

                newPerson.crme_ReturnMessage = response.MVIMessage + response.CORPDbMessage;

                Logger.WriteDebugMessage("DEBUG::Person has NOTHING: " + newPerson.crme_ReturnMessage);
                if (!string.IsNullOrEmpty(response.RawMviExceptionMessage))     {
                    Logger.WriteDebugMessage(response.RawMviExceptionMessage);
                }
                personCollection.Entities.Add(newPerson);

                PluginExecutionContext.OutputParameters["BusinessEntityCollection"] = personCollection;
                return;
            }

            string url = "";
            string vbaSensitiveLevel = "";
            if (getSensitiveInfo)  {
                // get the sensitivity endpoint
                url = getKVPSetting("sensitive_endpoint");              
            }        
   
            // for each person returned create a Person in CRM
            foreach (var person in response.Person)  {
                #region map the person with data
                newPerson = new crme_person();
                newPerson.Id = Guid.NewGuid();
                newPerson.crme_personId = newPerson.Id;

                if (setMessage)    {
                    setMessage = false;
                    newPerson.crme_ExceptionMessage = "Your search in MVI did not find any records matching the search criteria.";
                }

                TryGetMviQueryParams(person, newPerson, Logger);

                // need to handle VBA sensitivity for each result
                if (orgName.ToUpper().Contains("VRE") && getSensitiveInfo)      {
                    getMVICorpPidBIRLSFN(newPerson);

                    bool usedPid = true;
                    if (newPerson.crme_ParticipantID == "" && newPerson.crme_FileNumber != "")  {
                        vbaSensitiveLevel = getSensitivityLevelVBA(url, newPerson.crme_FileNumber, false);      // sending false will search by FN and not Pid
                        usedPid = false;
                    }
                    else if (newPerson.crme_ParticipantID != "")
                        vbaSensitiveLevel = getSensitivityLevelVBA(url, newPerson.crme_ParticipantID, true);

                    // check if we got a value with one, if not try the other
                    if (vbaSensitiveLevel == "")      {
                        if (usedPid)     {
                            vbaSensitiveLevel = getSensitivityLevelVBA(url, newPerson.crme_FileNumber, false);   
                        }
                        else
                            vbaSensitiveLevel = getSensitivityLevelVBA(url, newPerson.crme_ParticipantID, true);
                    }

                    if (vbaSensitiveLevel == "")
                        vbaSensitiveLevel = "0";
                    
                    newPerson.crme_VeteranSensitivityLevel = vbaSensitiveLevel;
                    int userLvl = Convert.ToInt32(userSensLevel);
                    int callerLvl = Convert.ToInt32(vbaSensitiveLevel);

                    Logger.WriteDebugMessage("LOG::User Level: " + userLvl + ": Caller Level: " + callerLvl);

                    if (userLvl < callerLvl)     { // user does not have access     
                        // log attempted access
                        auditSensitivity(person.FullName, false);

                        newPerson.crme_FirstName = "Sensitive";
                        newPerson.crme_LastName = "Record";
                        continue;
                    }
                }
                else    {
                    TryGetCorpInfo(person, newPerson, Logger);
                    newPerson.crme_VeteranSensitivityLevel = person.VeteranSensitivityLevel;
                    newPerson.crme_EDIPI = person.EdiPi;
                }
                          

                try
                {
                    if (person.NameList != null)    {
                        var legalName = person.NameList.FirstOrDefault(v => v.NameType.Equals("Legal", StringComparison.OrdinalIgnoreCase));
                        var alias = person.NameList.FirstOrDefault(v => v.NameType.Equals("Alias", StringComparison.OrdinalIgnoreCase));

                        if (legalName != null)     {
                            newPerson.crme_FirstName = legalName.GivenName;
                            newPerson.crme_LastName = legalName.FamilyName;
                            newPerson.crme_MiddleName = legalName.MiddleName;
                        }
                        else    {
                            legalName = person.NameList.FirstOrDefault();

                            if (legalName != null)    {
                                newPerson.crme_FirstName = legalName.GivenName;
                                newPerson.crme_LastName = legalName.FamilyName;
                                newPerson.crme_MiddleName = legalName.MiddleName;
                            }
                        }

                        newPerson.crme_Alias = TryGetAlias(alias);
                    }
                    
                    newPerson.crme_FullName = "";
                    newPerson.crme_FullName = FullNameBuilder(newPerson.crme_FirstName, newPerson.crme_MiddleName, newPerson.crme_LastName);

                    if (person.Address != null && person.Address.StreetAddressLine != "" && person.Address.State != "" && person.Address.City != "" && person.Address.State != "" && person.Address.PostalCode != "")    { 
                        newPerson.crme_FullAddress = person.Address.StreetAddressLine + " " + person.Address.City + " " + person.Address.State + " " + person.Address.PostalCode;         
                    }
                    else
                        newPerson.crme_FullAddress = person.FullAddress;

                    newPerson.crme_BranchOfService = person.BranchOfService;
                    newPerson.crme_PrimaryPhone = person.PhoneNumber;
                    newPerson.crme_RecordSource = person.RecordSource;
                    newPerson.crme_Gender = person.GenderCode;
                    newPerson.crme_DeceasedDate = person.DeceasedDate;
                    newPerson.crme_IdentityTheft = person.IdentifyTheft;
                    newPerson.crme_url = person.Url;
                    newPerson.crme_ReturnMessage = response.MVIMessage + response.CORPDbMessage;
                    //added from FtP, works with assisting in clicking on the record for the 2nd (selectedpersonsearch) MVI call
                    newPerson.crme_url = "&select=*&$filter=crme_PatientMviIdentifier eq '" + newPerson.crme_PatientMviIdentifier + "' and crme_SearchType eq 'SelectedPersonSearch'";

                    // will either set or mask the SSN and DOB values
                    HandleSensitivityMasking(person, newPerson, url);
                    
                    // handle auditing of all calls
                    if (enableAuditing) 
                        auditPersonResponse(newPerson);                    
                    
                    personCollection.Entities.Add(newPerson);
                    
                }
                catch (Exception e)
                {
                    //TODO Should this exception be handeled?               
                    Logger.WriteDebugMessage("ERROR::Error while mapping person from NameList: " + e.Message);
                }
                #endregion                                        
            }

            PluginExecutionContext.OutputParameters["BusinessEntityCollection"] = personCollection;         
         
        }

        /// <summary>
        /// This basically makes the 2nd MVI call to get the correlations to 
        /// get the FileNumber so we can get the sensitivity level and compare
        /// </summary>
        /// <param name="ICN"></param>
        /// <returns></returns>
        private void getMVICorpPidBIRLSFN(crme_person newPerson)
        {
            var selectedRequest = new SelectedPersonRequest();        
            selectedRequest.OrganizationName = orgName;
            selectedRequest.UserId = PluginExecutionContext.InitiatingUserId;
            selectedRequest.MessageId = Guid.NewGuid().ToString();
            selectedRequest.Debug = false;
            selectedRequest.LogSoap = _logSoap;
            selectedRequest.LogTiming = _logTimer;
            selectedRequest.noAddPerson = false;
            selectedRequest.ICN = newPerson.crme_ICN;

            var selectedResponse = Utility.SendReceive<CorrespondingIdsResponse>(uri, "VRESelectedPersonRequest", selectedRequest, _logSettings);

            if (selectedResponse.ExceptionOccured)   {
                Logger.WriteDebugMessage(string.Format("ERROR::VBASensitivityCheck: Search Exception Message: {0}", selectedResponse.RawMviExceptionMessage));
            }
            //else
            //    Logger.WriteDebugMessage(string.Format("LOG::VBASensitivityCheck: Success: {0}", selectedResponse.RawMviExceptionMessage));

            try
            {
                if (selectedResponse.CorrespondingIdList == null || !selectedResponse.CorrespondingIdList.Any())
                {
                    return;
                }
                else
                {
                    var patientNo = selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                        v.AssigningAuthority != null &&
                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                        v.AssigningFacility != null && v.AssigningFacility == "200CORP" &&
                        v.IdentifierType != null &&
                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase))) ??
                                    selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                                        v.AssigningAuthority != null &&
                                        v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                                        v.IdentifierType != null &&
                                        v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase)));
                    newPerson.crme_ParticipantID = patientNo != null ? patientNo.PatientIdentifier : string.Empty;

                    var patientFN = selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                      v.AssigningAuthority != null &&
                      v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                      v.AssigningFacility != null && v.AssigningFacility == "200BRLS" &&
                      v.IdentifierType != null &&
                      v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase))) ??
                                  selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                                      v.AssigningAuthority != null &&
                                      v.AssigningAuthority.Equals("USVBA", StringComparison.InvariantCultureIgnoreCase) &&
                                      v.IdentifierType != null &&
                                      v.IdentifierType.Equals("PI", StringComparison.InvariantCultureIgnoreCase)));                   
                    newPerson.crme_FileNumber = patientFN != null ? patientFN.PatientIdentifier : string.Empty;

                    //999872003^NI^200DOD^USDOD}
                    var patientEDIPI = selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                     v.AssigningAuthority != null &&
                     v.AssigningAuthority.Equals("USDOD", StringComparison.InvariantCultureIgnoreCase) &&
                     v.AssigningFacility != null && v.AssigningFacility == "200DOD" &&
                     v.IdentifierType != null &&
                     v.IdentifierType.Equals("NI", StringComparison.InvariantCultureIgnoreCase))) ??
                                 selectedResponse.CorrespondingIdList.FirstOrDefault((v =>
                                     v.AssigningAuthority != null &&
                                     v.AssigningAuthority.Equals("USDOD", StringComparison.InvariantCultureIgnoreCase) &&
                                     v.IdentifierType != null &&
                                     v.IdentifierType.Equals("NI", StringComparison.InvariantCultureIgnoreCase)));

                    newPerson.crme_EDIPI = patientEDIPI != null ? patientEDIPI.PatientIdentifier : string.Empty;

                    Logger.WriteDebugMessage(string.Format("LOG::VBASensitivityCheck: Response Success for PId: {0}, FileNumber: {1}, EDIPI: {2} ", patientNo.PatientIdentifier, patientFN.PatientIdentifier, patientEDIPI.PatientIdentifier));
                }
            }
            catch (Exception ex)
            {
                  Logger.WriteDebugMessage("ERROR::VBASensitivityCheck: Error in getMVICorpPidBIRLSFN()");
            }
        }

        //private void Map(PersonSearchResponse response)
        //{
        //    var personCollection = new EntityCollection();
        //    crme_person newPerson = null;
        //    bool setMessage = false;

        //    // exception check
        //    if (response.ExceptionOccured)
        //    {
        //        if (response.MVIMessage.ToLower().Contains("correlation does not exist") || response.MVIMessage.ToLower().Contains("unknown key identifier"))
        //        {
        //            // Not a real exception. The EDIPI not found throws an application exception in MVI but we just want to report Veteran not found
        //            setMessage = true;
        //        }
        //        else
        //        {
        //            newPerson = new crme_person { Id = Guid.NewGuid(), crme_FullName = string.Empty };
        //            newPerson.crme_ExceptionOccured = true;
        //            newPerson.crme_ExceptionMessage = "An unexpected error occured during the MVI search. Please try again or contact your system administrator if the problem persists.";
        //        }
        //    }

        //    if (response.Person == null || (response.Person != null && response.Person[0] == null))
        //    {
        //        if (newPerson == null)
        //        {
        //            newPerson = new crme_person { Id = Guid.NewGuid(), crme_FullName = string.Empty };
        //        }

        //        if (setMessage)
        //        {
        //            newPerson.crme_ExceptionMessage = "Your search in MVI did not find any records matching the search criteria.";
        //        }

        //        newPerson.crme_ReturnMessage = response.MVIMessage + response.CORPDbMessage;

        //        Logger.WriteDebugMessage("DEBUG::Person has NOTHING: " + newPerson.crme_ReturnMessage);
        //        if (!string.IsNullOrEmpty(response.RawMviExceptionMessage))
        //        {
        //            Logger.WriteDebugMessage(response.RawMviExceptionMessage);
        //        }
        //        personCollection.Entities.Add(newPerson);

        //        PluginExecutionContext.OutputParameters["BusinessEntityCollection"] = personCollection;
        //        return;
        //    }

        //    string url = "";
        //    if (getSensitiveInfo)
        //    {
        //        // get the sensitivity endpoint
        //        url = getKVPSetting("sensitive_endpoint");
        //    }

        //    // for each person returned create a Person in CRM
        //    foreach (var person in response.Person)
        //    {
        //        #region map the person with data
        //        newPerson = new crme_person();
        //        newPerson.Id = Guid.NewGuid();
        //        newPerson.crme_personId = newPerson.Id;

        //        if (setMessage)
        //        {
        //            setMessage = false;
        //            newPerson.crme_ExceptionMessage = "Your search in MVI did not find any records matching the search criteria.";
        //        }

        //        // need to handle VBA sensitivity for each result
        //        if (orgName.ToUpper().Contains("VRE"))
        //        {
        //            // need to get the Corp Participant Id and check access level before returning the record



        //        }


        //        try
        //        {
        //            if (person.NameList != null)
        //            {
        //                var legalName = person.NameList.FirstOrDefault(v => v.NameType.Equals("Legal", StringComparison.OrdinalIgnoreCase));
        //                var alias = person.NameList.FirstOrDefault(v => v.NameType.Equals("Alias", StringComparison.OrdinalIgnoreCase));

        //                if (legalName != null)
        //                {
        //                    newPerson.crme_FirstName = legalName.GivenName;
        //                    newPerson.crme_LastName = legalName.FamilyName;
        //                    newPerson.crme_MiddleName = legalName.MiddleName;
        //                }
        //                else
        //                {
        //                    legalName = person.NameList.FirstOrDefault();

        //                    if (legalName != null)
        //                    {
        //                        newPerson.crme_FirstName = legalName.GivenName;
        //                        newPerson.crme_LastName = legalName.FamilyName;
        //                        newPerson.crme_MiddleName = legalName.MiddleName;
        //                    }
        //                }

        //                newPerson.crme_Alias = TryGetAlias(alias);
        //            }

        //            newPerson.crme_FullName = "";
        //            newPerson.crme_FullName = FullNameBuilder(newPerson.crme_FirstName, newPerson.crme_MiddleName, newPerson.crme_LastName);

        //            if (person.Address != null && person.Address.StreetAddressLine != "" && person.Address.State != "" && person.Address.City != "" && person.Address.State != "" && person.Address.PostalCode != "")
        //            {
        //                newPerson.crme_FullAddress = person.Address.StreetAddressLine + " " + person.Address.City + " " + person.Address.State + " " + person.Address.PostalCode;
        //            }
        //            else
        //                newPerson.crme_FullAddress = person.FullAddress;

        //            newPerson.crme_BranchOfService = person.BranchOfService;
        //            newPerson.crme_PrimaryPhone = person.PhoneNumber;
        //            newPerson.crme_RecordSource = person.RecordSource;
        //            newPerson.crme_Gender = person.GenderCode;
        //            newPerson.crme_DeceasedDate = person.DeceasedDate;
        //            newPerson.crme_IdentityTheft = person.IdentifyTheft;
        //            newPerson.crme_url = person.Url;
        //            newPerson.crme_ReturnMessage = response.MVIMessage + response.CORPDbMessage;
        //            newPerson.crme_EDIPI = person.EdiPi;
        //            newPerson.crme_ParticipantID = person.ParticipantId;
        //            newPerson.crme_VeteranSensitivityLevel = person.VeteranSensitivityLevel;

        //            //added from FtP, works with assisting in clicking on the record for the 2nd (selectedpersonsearch) MVI call
        //            newPerson.crme_url = "&select=*&$filter=crme_PatientMviIdentifier eq '" + newPerson.crme_PatientMviIdentifier + "' and crme_SearchType eq 'SelectedPersonSearch'";

        //            TryGetMviQueryParams(person, newPerson, Logger);
        //            TryGetCorpInfo(person, newPerson, Logger);
        //            TryGetEDIPI(person, newPerson, Logger);

        //            // will either set or mask the SSN and DOB values
        //            HandleSensitivity(person, newPerson, url);

        //            // handle auditing of all calls
        //            if (enableAuditing)
        //                auditPersonResponse(newPerson);

        //            personCollection.Entities.Add(newPerson);
        //        }
        //        catch (Exception e)
        //        {
        //            //TODO Should this exception be handeled?               
        //            Logger.WriteDebugMessage("ERROR::Error while mapping person from NameList: " + e.Message);
        //        }
        //        #endregion
        //    }

        //    PluginExecutionContext.OutputParameters["BusinessEntityCollection"] = personCollection;
        //}

        private void HandleSensitivityMasking(PatientPerson person, crme_person newPerson, string url)
        {
            // check the ICN in ESR and set sensitivity level to verify if masking is needed
            if (getSensitiveInfo && newPerson.crme_ICN != "" && url != "" && !orgName.ToUpper().Contains("VRE"))
            {
                newPerson.crme_VeteranSensitivityLevel = getSensitivityLevelVHA(url, newPerson.crme_ICN);

                // handle masking of SSN and DOB based on sensitivity and veteran indicators, here mask if sensitive = true
                if (orgName.ToUpper().Contains("VCL") && (newPerson.crme_VeteranSensitivityLevel == "true:false" || newPerson.crme_VeteranSensitivityLevel == "true:true"))
                {
                    newPerson.crme_DOBString = "XX/XX/XXXX";
                    newPerson.crme_SSN = "XXXXXXXXX";
                }
                else if (newPerson.crme_VeteranSensitivityLevel == "true:true" || newPerson.crme_VeteranSensitivityLevel == "false:true") //mask if Employee = true (2nd value)
                {
                    newPerson.crme_DOBString = "XX/XX/XXXX";
                    newPerson.crme_SSN = "XXXXXXXXX";
                }
                else
                {
                    newPerson.crme_DOBString = person.BirthDate;
                    newPerson.crme_SSN = person.SocialSecurityNumber;
                }
            }
            else
            {
                newPerson.crme_DOBString = person.BirthDate;
                newPerson.crme_SSN = person.SocialSecurityNumber;
            }
        }

        /// <summary>
        /// Will return a string of the response from MVI
        /// </summary>
        /// <param name="count"></param>
        /// <param name="person"></param>
        /// <returns></returns>
        private void auditPersonResponse(crme_person person)
        {
            if (auditUrl == "")
                auditUrl = getKVPSetting("MVI_Log");

            string tempICN = person.crme_ICN;
            if (tempICN.Length > 15)
                tempICN = person.crme_ICN.Substring(0, 15);

            var MviLog = new MVILog   {
                Address = person.crme_FullAddress,
                Attended = person.crme_IsAttended == null ? false : (bool)person.crme_IsAttended,
                ClassCode = person.crme_ClassCode,
                CreatedOn = DateTime.UtcNow,
                DOB = person.crme_DOBString,
                EDIPI = person.crme_EDIPI,
                First = person.crme_FirstName,
                ICN = tempICN,
                Last = person.crme_LastName,
                Middle = person.crme_MiddleName,
                MMN = person.crme_MMN,
                Organization_Name = orgName,
                Phone = person.crme_PrimaryPhone,
                POBC = person.crme_POBC,
                POBS = person.crme_POBS,
                RequestType = person.crme_SearchType,
                SessionId = new Guid().ToString(),
                Sex = person.crme_Gender,
                Source = person.crme_RecordSource,
                Type = "Response",
                UserName = userName,
                SSN = person.crme_SSN
            };

            var json = new JavaScriptSerializer().Serialize(MviLog);
            handleRequest(auditUrl, json, "MVILog");
        }

        /// <summary>
        /// Handles sending the requests and responses for audit logging and sensitivity
        /// </summary>
        /// <param name="url">The url of the service</param>
        /// <param name="json">the json string of the object</param>
        private void handleRequest(string url, string json, string logTable)
        {
            var httpWebRequest = (HttpWebRequest)WebRequest.Create(auditUrl);
            httpWebRequest.ContentType = "application/json";
            httpWebRequest.Method = "POST";

            using (var streamWriter = new StreamWriter(httpWebRequest.GetRequestStream()))
            {
                streamWriter.Write(json);
                streamWriter.Flush();
                streamWriter.Close();
            }

            try
            {                
                var httpResponse = (HttpWebResponse)httpWebRequest.GetResponse();
                using (var streamReader = new StreamReader(httpResponse.GetResponseStream()))
                {
                    streamReader.ReadToEnd();
                    streamReader.Close();
                }                
                httpResponse.Close();
            }
            catch (Exception e)
            {
                Logger.WriteDebugMessage("ERROR::Error creating " + logTable + " response record: " + e.Message);
            }
        }

        private string FullNameBuilder(string first, string middle, string last)
        {
            string fullName = "n/a";

            var fName = (string.IsNullOrEmpty(first)) ? "" : first;
            var mName = (string.IsNullOrEmpty(middle)) ? "" : middle;
            var lName = (string.IsNullOrEmpty(last)) ? "" : last;

            fullName = string.Format("{0} {1} {2}", fName, mName, lName);

            return fullName;
        }

        private void HandleMVIErrorInUI()
        {
            // need to check if there has already been a person added to the entity collection or not
            if (((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities != null &&
                ((EntityCollection)PluginExecutionContext.OutputParameters["BusinessEntityCollection"]).Entities.Count > 0)
            {

            }

            var personCollection = new EntityCollection();

            // var newPerson = new crme_person();
            crme_person newPerson = new crme_person();
            newPerson.crme_personId = Guid.NewGuid();
            newPerson.crme_FullName = string.Empty;
            newPerson.crme_ExceptionOccured = true;
            newPerson.crme_ExceptionMessage = "An unexpected error occured during the MVI search. Please try again or contact your system administrator if the problem persists.";

            Logger.WriteDebugMessage("ERROR::Sending back empty Person for Exception Handling.");
            personCollection.Entities.Add(newPerson);

            PluginExecutionContext.OutputParameters["BusinessEntityCollection"] = personCollection;
            return;
        }

        private string FormatDate(string date)
        {
            if (date != "" && !string.IsNullOrWhiteSpace(date) && date.Length == 8)
            {
                return date.Substring(4, 2) + "/" + date.Substring(6, 2) + "/" + date.Substring(0, 4);
            }

            return date;
        }

        #endregion
    }

    /// <summary>
    /// 
    /// </summary>
    //internal class User
    //{
    //    public string FirstName { get; set; }
    //    public string LastName { get; set; }
    //}

    internal class SensitiveLog
    {
        public string Organization_Name;
        public string Veteran;
        public bool Sensitive_Veteran;
        public bool Accessed;
        public string Who_Accessed;
        public string HDR_Response_Code;
        public string Access_Time;
        public string SessionId;

    }

    internal class MVILog
    {
        public string Organization_Name;
        public string First;
        public string Last;
        public string Middle;
        public string DOB;
        public string SSN;
        public string EDIPI;
        public string Phone;
        public bool Attended; 
        public string Address;
        public string Sex;
        public string MMN;
        public string POBC;
        public string POBS;
        public string RequestType;
        public string ICN; //15
        public string Source;
        public string ClassCode;
        public string Type;
        public DateTime CreatedOn;
        public string SessionId;
        public string UserName;
    }

    internal class MVILog2
    {
        public string Organization_Name;
        public string DOB;
        public string SSN;
        public string EDIPI;
        public string Phone;
        public string ICN;
        public string IDs;
        public DateTime CreatedOn;
        public string SessionId;
        public string UserName;
        public string RequestType;
    }

    internal class SensitivityInfo
    {
        public string SensityFlag { get; set; }
    }

    internal class PrimaryEligibility
    {
        public string Type { get; set; }
    }

    internal class EnrollmentDeterminationInfo
    {
        public PrimaryEligibility PrimaryEligibility { get; set; }
        public string Veteran { get; set; }
    }

    internal class Demographics
    {
        public string PreferredFacility { get; set; }
    }

    internal class Data
    {
        public SensitivityInfo SensitivityInfo { get; set; }
        public EnrollmentDeterminationInfo EnrollmentDeterminationInfo { get; set; }
        public Demographics Demographics { get; set; }
    }

    internal class ESRSensitiveObject
    {
        public bool ErrorOccurred { get; set; }
        public object ErrorMessage { get; set; }
        public object Status { get; set; }
        public object DebugInfo { get; set; }
        public Data Data { get; set; }
    }

    public class SensitiveInfo
    {
        public string BdnStnNbr { get; set; }
        public string BirlsFolderLctnTxt { get; set; }
        public string Cd { get; set; }
        public string DgnstcSuprsnInd { get; set; }
        public string FcltyTypeCd { get; set; }
        public DateTime JrnDt { get; set; }
        public string JrnLctnId { get; set; }
        public string JrnObjId { get; set; }
        public string JrnStatusTypeCd { get; set; }
        public string JrnUserId { get; set; }
        public string PtcpntId { get; set; }
        public string RmksTxt { get; set; }
        public string ScrtyLevelTypeCd { get; set; }
        public string SntvtyLevelId { get; set; }
    }

    public class Datum
    {
        public object Fault { get; set; }
        public SensitiveInfo Return1 { get; set; }
    }

    public class SensitiveObject
    {
        public bool ErrorOccurred { get; set; }
        public object ErrorMessage { get; set; }
        public object Status { get; set; }
        public object DebugInfo { get; set; }
        public List<Datum> Data { get; set; }
    }
}